
#include "dd_spi.h"

void (*spi_callback)(unsigned char, unsigned char);  //This is a pointer to the function which 
                                                     //the user needs to pass in SCI_Init
                                                     //so that the required function is 
                                                     //called in interrupt subroutine

unsigned char SPI0_Data;     //variable to store the data for SPI0 to be transmitted if interrupt is enabled
unsigned char SPI1_Data;     //variable to store the data for SPI0 to be transmitted if interrupt is enabled
unsigned char SPI2_Data;     //variable to store the data for SPI0 to be transmitted if interrupt is enabled

/******************************************************************************/
/* FUNCTION          : SPI_Init                                               */
/******************************************************************************/
/* Abstract          : This function initializes Specified SPI by configuring */
/*                     the internal registers                                 */
/*					                                                                  */
/*                                                                            */
/* Input Parameters  : SPI_Index :SPI_Index0/SPI_Index1/SPI_Index2            */ 
/*                     Master_Slave_Mode :SPI_SLAVE_MODE/SPI_MASTER_MODE      */
/*                     Configure_Slave_Select_Pin:SPI_SS_PIN_GPIO             */
/*                                                /SPI_SS_PIN_SS_OUTPUT       */
/*                                                /SPI_SS_PIN_MODE_FAULT      */
/*                     Clock_Polarity :SPI_CLOCK_POL_0/SPI_CLOCK_POL_1        */
/*                     Clock_Phase    :SPI_CLOCK_PHASE_0/SPI_CLOCK_PHASE_1    */
/*                     p : pointer to the function passed.User needs to pass  */
/*                         the address of the callback function needed in case*/
/*                         of interrupt                                       */                                                             
/*                                                                            */
/* Return Parameter  :                                                        */
/******************************************************************************/

void SPI_Init(unsigned char SPI_Index,unsigned char Master_Slave_Mode,
              unsigned char Configure_Slave_Select_Pin,unsigned char Clock_Polarity,
              unsigned char Clock_Phase,void (*p)(unsigned char, unsigned char)) 
{ 
  SPI *spi_ptr;
  SCGC4_MUXCTRL=1;                 //enables the clock to the pin mux module

  spi_callback = p;
  switch(SPI_Index) 
  {
      case 0:                     //SPI0 selected
        {
          SCGC2_SPI0 = 1;           //enabling the clock to SPI0   
          spi_ptr = (SPI*)SPI_0;    //points to SPI0
          //Pin mux
          PTBPF2_B2 = 2;            //Pin PTB2 as MOSI0
          PTBPF2_B3 = 2;            //Pin PTB3 as MISO0
          PTBPF3_B4 = 2;            //Pin PTB4 as SCLK0
          PTBPF3_B5 = 2;            //Pin PTB5 as SS0
          break;
        }
      
      case 1:                    //SPI1 selected
        {
          SCGC2_SPI1 =1;           //enabling the clock for SPI1
          spi_ptr = (SPI*)SPI_1;   //points to SPI1
          //Pin Mux
          #if SPI1_Pins_PortC 
            {
              PTCPF1_C0 = 1;         //Pin PTC0 as MOSI1
              PTCPF1_C1 = 1;         //Pin PTC1 as MISO1
              PTCPF2_C2 = 1;         //Pin PTC2 as SCLK1
              PTCPF2_C3 = 1;         //Pin PTC3 as SS1
            }
          #else 
            {
              PTGPF1_G0 = 1;         //Pin PTG0 as MOSI1
              PTGPF1_G1 = 1;         //Pin PTG1 as MISO1
              PTGPF2_G2 = 1;         //Pin PTG2 as SCLK1
              PTGPF2_G3 = 1;         //Pin PTG3 as SS1
            }
          #endif 
          break;
        }
    
      case 2:                    //SPI2 selected
        {  
          SCGC2_SPI2 =1;            //enabling the clock to SP12
          spi_ptr = (SPI*)SPI_2;    //points to SPI2                      
          //Pin mux
          #if SPI2_Pins_PortA 
            {  
              PTAPF1_A0 = 1;         //Pin PTA0 as MOSI2
              PTAPF1_A1 = 1;         //Pin PTA1 as MISO2
              PTAPF2_A2 = 1;         //Pin PTA2 as SCLK2
              PTAPF2_A3 = 1;         //Pin PTA3 as SS2
            }
          #else 
            {
              PTDPF1_D0 = 2;         //Pin PTD0 as MOSI2
              PTDPF1_D1 = 2;         //Pin PTD1 as MISO2
              PTDPF2_D2 = 2;         //Pin PTD2 as SCLK2
              PTDPF2_D3 = 2;         //Pin PTD3 as SS2
            }
          #endif
          break;
        }
  }
  
  /* Set the Baud Rate*/
  /* Baud rate can be calculated using the formula
     
     SPI_CLK = Bus_Clk /(Prescaler_Divider * Divider)
   
     SPPR2:SPPR1:SPPR0      Prescaler Divisor
           0:0:0                  1
           0:0:1                  2
           0:1:0                  3
           0:1:1                  4
           1:0:0                  5
           1:0:1                  6
           1:1:0                  7
           1:1:1                  8
           
      SPR3:SPR2:SPR1:SPR0         Rate Divisor
           0:0:0:0                  2
           0:0:0:1                  4
           0:0:1:0                  8
           0:0:1:1                  16
           0:1:0:0                  32
           0:1:0:1                  64
           0:1:1:0                  128
           0:1:1:1                  256      
           1:0:0:0                  512
           other combinations       reserved
  */
  
  spi_ptr->SPIBR.Byte = 0x43;        //sets the baud rate to 250KHz
  
  /*
                      *  0b01000011
                      *    ||||||||__ SPR0
                      *    |||||||___ SPR1
                      *    ||||||____ SPR2
                      *    |||||_____ SPR3
                      *    ||||______ SPPR0
                      *    |||_______ SPPR1
                      *    ||________ SPPR2
                      *    |_________ Unimplemented
                      */
   /* 
   Thus the baud rate set is 
   BUS_CLK/(prescaler divider * rate divisor)
   = 20,000,000 /(5*16) = 250KHz  
   */
  spi_ptr->SPIC1.Bits.MSTR = Master_Slave_Mode;    //selects between master and slave mode
  
  if(spi_ptr->SPIC1.Bits.MSTR)  //if configured as master
  {
    switch(Configure_Slave_Select_Pin) 
      {
        case SPI_SS_PIN_GPIO:                            //SS pin configured as GPIO
              spi_ptr->SPIC2.Bits.MODFEN = 0;
              spi_ptr->SPIC1.Bits.SSOE   = 0;
              break;
              
        case SPI_SS_PIN_MODE_FAULT:                      //SS pin configured as Input for mode fault
              spi_ptr->SPIC2.Bits.MODFEN = 1;
              spi_ptr->SPIC1.Bits.SSOE   = 0;
              break;
                
        case SPI_SS_PIN_SS_OUTPUT:                       //SS pin configured as automatic SS output
              spi_ptr->SPIC2.Bits.MODFEN = 1;
              spi_ptr->SPIC1.Bits.SSOE   = 1;
              break;
      }
  }
  
  /*if configured as slave then the SS pin always act as SS input*/
  
  spi_ptr->SPIC1.Bits.CPOL = Clock_Polarity;         //sets the clock polarirty 
  spi_ptr->SPIC1.Bits.CPHA = Clock_Phase;            //sets the clock phase
  
  spi_ptr->SPIC1.Bits.SPE = 1;                       //enables the SPI 
}
 
/******************************************************************************/
/* FUNCTION          : SPI_Set_Baud_Rate                                      */
/******************************************************************************/
/* Abstract          : This function sets the Baud Rate of the specific SPI   */
/*					                                                                  */
/*                                                                            */
/* Input Parameters  : SPI_Index :SPI_Index0/SPI_Index1/SPI_Index2            */ 
/*                     Baud_Rate :The baud rate (in KHz)                      */
/*                     eg: if a bauud rate of 250 khz is to be set for SPI0   */
/*                         SPI_Set_Baud_Rate(SPI_Index0,250)                  */
/*                                                                            */
/* Return Parameter  :                                                        */
/******************************************************************************/
   
unsigned char SPI_Set_Baud_Rate(unsigned char SPI_Index,unsigned int Baud_Rate) //rate in Khz
{
 unsigned int value;
 unsigned char i=0;
 SPI *spi_ptr;
 switch(SPI_Index) 
  {
    case 0:
      spi_ptr = (SPI*)SPI_0;    //points to SPI0
      break;
    
    case 1:
      spi_ptr = (SPI*)SPI_1;    //points to SPI1
      break;
      
    case 2:
      spi_ptr = (SPI*)SPI_2;    //points to SPI2
      break;
  } 
 
 spi_ptr->SPIBR.Byte = 0;
 value = (BUS_CLK /Baud_Rate);           //baud rate = BUS_CLK / (prescaler divisor * rate divisor)
  
 while(((value%2) == 0)) 
  {
    value=(value/2);                      //calculates the power of 2 in the baud rate divisor
    i++;
    if(i==9)
    break;
  }
 if(i>=1)
  { 
    spi_ptr->SPIBR.Byte =((spi_ptr->SPIBR.Byte) | (i-1));    //sets the rate divisor in the bayd rate register
  }
 
 for(i=8;i>=1;i--)
  {
    if((value%i)== 0) 
      {
        value = value/i;
        spi_ptr->SPIBR.Byte =((spi_ptr->SPIBR.Byte)| ((i-1)<<4)); //sets the prescaler divisor in the baud rate register
        if(value != 1) 
          {
            return 0;     //The baud rate value to be set is not a valid value
            }
        break;
      }
  }
 return 1;         //Baud Rate successfully set 
}


/******************************************************************************/
/* FUNCTION          : SPI_Write_Char                                         */
/******************************************************************************/
/* Abstract          : This function sends a char to the SPI port             */
/*					                                                                  */
/*                                                                            */
/* Input Parameters  : SPI_Index :SPI_Index0/SPI_Index1/SPI_Index2            */ 
/*                     Data : data to be sent                                 */
/*                     Interrupt_Enable :SPI_INTERRUPT_ENABLE                 */
/*                                       /SPI_INTERRUPT_DISABLE               */
/*                                                                            */
/* Return Parameter  :                                                        */
/******************************************************************************/

  
 
 void SPI_Write_Char(unsigned char SPI_Index, unsigned char Data,unsigned char Interrupt_Enable)
 {
  Send_Data(SPI_Index,Data,Interrupt_Enable);       //data is transmitted 
  Wait_For_Data(SPI_Index,Interrupt_Enable);       //waits for the ACK
 }

/******************************************************************************/
/* FUNCTION          : SPI_Read_Char                                          */
/******************************************************************************/
/* Abstract          : This function reads a byte from the  SPI port          */
/*					                                                                  */
/*                                                                            */
/* Input Parameters  : SPI_Index :SPI_Index0/SPI_Index1/SPI_Index2            */ 
/*                     Interrupt_Enable :SPI_INTERRUPT_ENABLE                 */
/*                                       /SPI_INTERRUPT_DISABLE               */
/*                                                                            */
/* Return Parameter  :                                                        */
/******************************************************************************/
 
 unsigned char SPI_Read_Char(unsigned char SPI_Index,unsigned char Interrupt_Enable) 
 {
  unsigned char data;
  Send_Data(SPI_Index,0xAA,Interrupt_Enable);           //A dummy data is transmitted to initiate the transfer
  data = Wait_For_Data(SPI_Index,Interrupt_Enable);    //Waiting for the data to be read
  return data;               //returns the data read
 }
 
/******************************************************************************/
/* FUNCTION          : Send_Data                                              */
/******************************************************************************/
/* Abstract          : This function sends char data to SPI port              */
/*                      (used in other functions only)                        */
/*                                                                            */
/* Input Parameters  : SPI_Index :SPI_Index0/SPI_Index1/SPI_Index2            */
/*                     Data : Data to be sent                                 */ 
/*                     Interrupt_Enable :SPI_INTERRUPT_ENABLE                 */
/*                                       /SPI_INTERRUPT_DISABLE               */
/*                                                                            */
/* Return Parameter  :                                                        */
/******************************************************************************/
 
 void Send_Data(unsigned char SPI_Index,unsigned char Data,unsigned char Interrupt_Enable)
 {
  SPI *spi_ptr;
  switch(SPI_Index) 
  {
    case 0:
    spi_ptr = (SPI*)SPI_0;
    SPI0_Data = Data;
    break;
    
    case 1:
    spi_ptr = (SPI*)SPI_1;
    SPI1_Data = Data;
    break;
    
    case 2:
    spi_ptr = (SPI*)SPI_2;
    SPI2_Data = Data;
  }
  
  if(Interrupt_Enable == SPI_INTERRUPT_ENABLE)
  {
   spi_ptr->SPIC1.Bits.SPIE = 1;        //enables the interrupt (SPRF - receive buffer full) & (MODF - Mode fault)
   spi_ptr->SPIC1.Bits.SPTIE = 1;  // interrupt enabled (SPTEF - SPI transmit buffer empty)
  }
  else  
  { 
   while(!(spi_ptr->SPIS.Bits.SPTEF));  //polling for the transmit buffer to be empty
   spi_ptr->SPID = Data;         //writes the data to the data reg 
  }
 }


/******************************************************************************/
/* FUNCTION          : Wait_For_Data                                          */
/******************************************************************************/
/* Abstract          : This function waits for data on the  SPI port          */
/*                      (used in other functions only)                        */
/*					                                                                  */
/*                                                                            */
/* Input Parameters  : SPI_Index :SPI_Index0/SPI_Index1/SPI_Index2            */ 
/*                     Interrupt_Enable :SPI_INTERRUPT_ENABLE                 */
/*                                       /SPI_INTERRUPT_DISABLE               */
/*                                                                            */
/* Return Parameter  :                                                        */
/******************************************************************************/

 unsigned char Wait_For_Data(unsigned char SPI_Index,unsigned char Interrupt_Enable) 
 {
  unsigned char data;
  SPI *spi_ptr;
  switch(SPI_Index) 
  {
    case 0:
    spi_ptr = (SPI*)SPI_0;
    break;
    
    case 1:
    spi_ptr = (SPI*)SPI_1;
    break;
    
    case 2:
    spi_ptr = (SPI*)SPI_2;
  }
  if(Interrupt_Enable == SPI_INTERRUPT_ENABLE)
  {
   while(spi_ptr->SPIC1.Bits.SPIE);    //waiting till the interrupt is not disabled 
   data = spi_ptr->SPID;
   spi_ptr->SPIC1.Bits.SPIE = 1;           //enabling the interrupt(SPRF - receive buffer full) & (MODF - Mode fault) 
   return data;
  }
  else 
  {  
   while(!(spi_ptr->SPIS.Bits.SPRF));  //Waiting the Receive buffer full flag to be set
  
   data = spi_ptr->SPID;         //Reads the data and clears the data register
   return data;
  }
 }
       
                           
/*Interrupt SubRoutine*/

//SPI0
void interrupt VectorNumber_Vspi0 SPI0_ISR() 
{
  unsigned char data;
  
  if(SPI0S & SPI0S_MODF_MASK)          // if MODE FAULT error detected
    {
      SPI0C1 = 0x04;                   // Write in SPIC1 to clear MODF
    }
  
  else if(SPI0S & SPI0S_SPRF_MASK)     //if SPRF is set 
    {
      SPI0C1_SPIE = 0;                 // Disable further Rx full interrupt requests
      //SPIC1_SPTIE = 0;               // Disable further Tx empty interrupt requests
      data = SPI0D;
    
      if(spi_callback != 0)
        (*spi_callback)(SPI0_RX,data);                                                         
    //sends the data to a function pointed by sci_callback
    }
    
  else if(SPI0S & SPI0S_SPTEF_MASK)    //if SPTEF is set 
    {
      SPI0C1_SPTIE = 0;                // disable further Tx empty interrupt requests
      SPI0D = SPI0_Data;
    }
}

//SPI1                           
void interrupt VectorNumber_Vspi1 SPI1_ISR() 
{
  unsigned char data;
  
  if(SPI1S & SPI1S_MODF_MASK)          // if MODE FAULT error detected
    {
      SPI1C1 = 0x04;                   // Write in SPIC1 to clear MODF
    }
  
  else if(SPI1S & SPI1S_SPRF_MASK)     //if SPRF is set 
    {
      SPI1C1_SPIE = 0;                 // Disable further Rx full interrupt requests
      //SPIC1_SPTIE = 0;               // Disable further Tx empty interrupt requests    
      data = SPI1D;
    
      if(spi_callback != 0)
        (*spi_callback)(SPI1_RX,data);                                                         
        //sends the data to a function pointed by sci_callback
    }

  else if(SPI1S & SPI1S_SPTEF_MASK)     //if SPTEF is set
    {
      SPI1C1_SPTIE = 0;                 // disable further Tx empty interrupt requests
      SPI1D = SPI1_Data;
    }
}                           

//SPI2
void interrupt VectorNumber_Vspi2 SPI2_ISR() 
{
  unsigned char data;
  
  if(SPI2S & SPI2S_MODF_MASK)           // if MODE FAULT error detected
    {
      SPI2C1 = 0x04;                    // Write in SPIC1 to clear MODF
    }
 
  else if(SPI2S & SPI2S_SPRF_MASK)      //if SPRF is set   
    {
      SPI2C1_SPIE = 0;                  // Disable further Rx full interrupt requests
      //SPIC1_SPTIE = 0;                // Disable further Tx empty interrupt requests
      data = SPI2D;
    
      if(spi_callback != 0)
        (*spi_callback)(SPI2_RX,data);                                                         
        //sends the data to a function pointed by sci_callback
    }
    
  else if(SPI2S & SPI2S_SPTEF_MASK)     //if SPTEF is set
    {
      SPI2C1_SPTIE = 0;                 // disable further Tx empty interrupt requests
      SPI2D = SPI2_Data;
    }
}